// RebeamformerTest.cpp : Defines the entry point for the console application.
//



#include "stdafx.h"
#include <fstream>
#include <iostream>
#include <vector>
#include <cstdlib>
 
#include <cstdint> 

#define _USE_MATH_DEFINES
#include <math.h> 
//#include "fft.h"
#include <ctime>

//PA Rebeamformer for simulated US Beamformed data.
//Input parameters include:
//ns,nl - number of samples and number of lines (depth and width of image)
//channelSpacing - spacing between transducer elements
//freq - frequency of sampling
//rf - input rf beamformed data.
//Outputs beamformed data in array (same dimensions as input array).

//Note, we precomputed the delays in the rfdelays and rfapts arrays. This increases runtime performance.
//max_samples defines the depth of the precomputed delay tables (set to 5000 samples per line to ensure all potential depths are precalculated).
int* pa_rebeamforming(const int ns, const int nl, int16_t* rf, int* rfdelays, int* rfapts, int max_samples) {
	int* postBF = new int[ns*nl];
	int* count = new int[ns*nl];

	for (int i = 0; i < ns*nl; i++) {
		postBF[i] = 0;
		count[i] = 1;
	}

	for (int r = 1; r <= nl; r++) {
		for (int j = 1; j <= ns; j++) {
			int half_apt = rfapts[j - 1];

			int imin = r - half_apt, imax = r + half_apt;
			if (imin < 1)
				imin = 1;
			if (imax > nl)
				imax = nl;

			for (int i = imin; i <= imax; i++) {
				int delay_pos = rfdelays[j-1 + abs(r-i)*max_samples];
					if (0 < delay_pos < ns) {
						postBF[j - 1 + (i - 1)*(ns)] = postBF[j - 1 + (i - 1)*(ns)] + (int)rf[delay_pos - 1 + (r - 1)*ns];
						count[j - 1 + (i - 1)*(ns)]++;
					}
			}
		}
	}

	for (int i = 0; i < nl*ns; i++)
		postBF[i] = postBF[i] / count[i];

	delete count;
	return postBF;
}




int main(int argc, char *argv[])
{

	double soundSpeed = 1480.0;  //1540
	double channelSpacing = 60.0 / 128.0; 
    double freq = 40e6;
	int lines = 128;//319
	std::string label;
	std::string outputlabel = "PA_Outputdata.txt";

	if (argc < 2)
		label = "RF_Data_Experiment.txt";
	else
		label = argv[1];

	std::ifstream ifile(label, std::ios::in);

	std::vector<int16_t> scores;



	//check to see that the file was opened correctly:
	if (!ifile.is_open()) {
		std::cout << "There was a problem opening the input file!\n";
		exit(1);//exit or do additional error checking
	}

	int count = 0;
    double num = 0.0;
	//keep storing values from the text file so long as data exists:
	while (ifile >> num) {
		scores.push_back(num);
		count++;
	}
	ifile.close();
	int sample = scores.size() / lines;


	
	static bool initialize = true;
	static int rfsamples = 5000;
	static int* rfdelay = new int[128 * rfsamples];

	//Precompute rfdelays and store in arrays.
	static int* rfapt = new int[rfsamples];
	if (initialize) {
		initialize = false;
		int line_num = 128;
		int sample_num = rfsamples;
		double sampleSpacing = 1 / freq * soundSpeed * 1000;
		for (int l = 0; l < line_num; l++)
			for (int s = 0; s < sample_num; s++) {
				double st_focus = (double)(s+1) / 2.0;

				double depth = ((double)s + 1.0 - st_focus)*(sampleSpacing);
				double width = ((double)l) * channelSpacing;
				double rad = sqrt((depth*depth) + width*width);
				double delay = st_focus + (rad) / (sampleSpacing);
				rfdelay[s + (l)*sample_num] = (delay + 0.5);
			}
		

		for (int s = 0; s < sample_num; s++) {
			double st_focus = (double)(s+1) / 2.0;
			rfapt[s] = (((((double)s + 1.0 - st_focus)*(sampleSpacing * 1.0) / channelSpacing)) + 0.5)/4; /// 8; /// 6;
		}
	}

	std::cout << "Data Loaded Into System. Beginning Beamforming.\n";
    std::clock_t start;
    double duration;

    start = std::clock();
	int* data = pa_rebeamforming(sample, lines,scores.data(), rfdelay, rfapt, rfsamples);
    //double** pdata = imageProcess(sample, lines, data);
    duration = (std::clock() - start) / (double)CLOCKS_PER_SEC;


	std::cout << "Beamforming Complete. Duration : " << duration << "\n";

	if (argc == 3)
		outputlabel = argv[2];

	std::ofstream ofile(outputlabel);
	//verify that the scores were stored correctly:
	for (int l = 0; l < lines; l++) {
		for (int s = 0; s < sample; s++) {
			ofile << data[l*sample + s] << ' ';
		}
		if(l != lines -1)
			ofile << '\n';
	}
	/*
	for (int i = 0; i < lines*sample; ++i) {
		ofile << scores[i] << std::endl;
	}*/
	ofile.close();


	std::cout << "ALL DONE :D \n";
	return 0;
}



